<?php

declare(strict_types=1);

namespace Square\Models;

use stdClass;

/**
 * Defines the fields that are included in the request body of a request
 * to the `CreateCustomerCard` endpoint.
 */
class CreateCustomerCardRequest implements \JsonSerializable
{
    /**
     * @var string
     */
    private $cardNonce;

    /**
     * @var Address|null
     */
    private $billingAddress;

    /**
     * @var string|null
     */
    private $cardholderName;

    /**
     * @var string|null
     */
    private $verificationToken;

    /**
     * @param string $cardNonce
     */
    public function __construct(string $cardNonce)
    {
        $this->cardNonce = $cardNonce;
    }

    /**
     * Returns Card Nonce.
     *
     * A card nonce representing the credit card to link to the customer.
     *
     * Card nonces are generated by the Square payment form when customers enter
     * their card information. For more information, see
     * [Walkthrough: Integrate Square Payments in a Website](https://developer.squareup.com/docs/web-
     * payments/take-card-payment).
     *
     * __NOTE:__ Card nonces generated by digital wallets (such as Apple Pay)
     * cannot be used to create a customer card.
     */
    public function getCardNonce(): string
    {
        return $this->cardNonce;
    }

    /**
     * Sets Card Nonce.
     *
     * A card nonce representing the credit card to link to the customer.
     *
     * Card nonces are generated by the Square payment form when customers enter
     * their card information. For more information, see
     * [Walkthrough: Integrate Square Payments in a Website](https://developer.squareup.com/docs/web-
     * payments/take-card-payment).
     *
     * __NOTE:__ Card nonces generated by digital wallets (such as Apple Pay)
     * cannot be used to create a customer card.
     *
     * @required
     * @maps card_nonce
     */
    public function setCardNonce(string $cardNonce): void
    {
        $this->cardNonce = $cardNonce;
    }

    /**
     * Returns Billing Address.
     *
     * Represents a postal address in a country.
     * For more information, see [Working with Addresses](https://developer.squareup.com/docs/build-
     * basics/working-with-addresses).
     */
    public function getBillingAddress(): ?Address
    {
        return $this->billingAddress;
    }

    /**
     * Sets Billing Address.
     *
     * Represents a postal address in a country.
     * For more information, see [Working with Addresses](https://developer.squareup.com/docs/build-
     * basics/working-with-addresses).
     *
     * @maps billing_address
     */
    public function setBillingAddress(?Address $billingAddress): void
    {
        $this->billingAddress = $billingAddress;
    }

    /**
     * Returns Cardholder Name.
     *
     * The full name printed on the credit card.
     */
    public function getCardholderName(): ?string
    {
        return $this->cardholderName;
    }

    /**
     * Sets Cardholder Name.
     *
     * The full name printed on the credit card.
     *
     * @maps cardholder_name
     */
    public function setCardholderName(?string $cardholderName): void
    {
        $this->cardholderName = $cardholderName;
    }

    /**
     * Returns Verification Token.
     *
     * An identifying token generated by [Payments.verifyBuyer()](https://developer.squareup.
     * com/reference/sdks/web/payments/objects/Payments#Payments.verifyBuyer).
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     */
    public function getVerificationToken(): ?string
    {
        return $this->verificationToken;
    }

    /**
     * Sets Verification Token.
     *
     * An identifying token generated by [Payments.verifyBuyer()](https://developer.squareup.
     * com/reference/sdks/web/payments/objects/Payments#Payments.verifyBuyer).
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     *
     * @maps verification_token
     */
    public function setVerificationToken(?string $verificationToken): void
    {
        $this->verificationToken = $verificationToken;
    }

    /**
     * Encode this object to JSON
     *
     * @param bool $asArrayWhenEmpty Whether to serialize this model as an array whenever no fields
     *        are set. (default: false)
     *
     * @return array|stdClass
     */
    #[\ReturnTypeWillChange] // @phan-suppress-current-line PhanUndeclaredClassAttribute for (php < 8.1)
    public function jsonSerialize(bool $asArrayWhenEmpty = false)
    {
        $json = [];
        $json['card_nonce']             = $this->cardNonce;
        if (isset($this->billingAddress)) {
            $json['billing_address']    = $this->billingAddress;
        }
        if (isset($this->cardholderName)) {
            $json['cardholder_name']    = $this->cardholderName;
        }
        if (isset($this->verificationToken)) {
            $json['verification_token'] = $this->verificationToken;
        }
        $json = array_filter($json, function ($val) {
            return $val !== null;
        });

        return (!$asArrayWhenEmpty && empty($json)) ? new stdClass() : $json;
    }
}
